% main_postestimation_simulations.m
% 
% post-estimation simulation exercises
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %



% -- Clear Workspace -- %
clc;
clear;
close all;
rng(61824)

% -- Options -- %
addpath('../Solver - SV')
addpath(genpath('../Utilities'));
GLOBALS = globals();


% -- Set up Table for Posterior Quantiles -- %
qm = getparams(GLOBALS);

% -- Solving the Model -- %
[xss,yss,sss,Pss,H,G,H2,G2,retcode] = endogenous_ms_solver_main(GLOBALS,qm);


%% -- SIMULATED DATA -- %
   
% data sets
simdatasets         = cell(GLOBALS.simnumber,1);
simregimesets       = cell(GLOBALS.simnumber,1);
simdatafull_shocks  = cell(GLOBALS.ne,1);
simdatasets_shocks  = cell(GLOBALS.ne,1);

for ii = 1:GLOBALS.simnumber    
    disp(ii)

    % -- Unconditional Simulations -- %
    [simdata,simregimes] = simulateddata_SV(GLOBALS,qm,Pss,xss,yss,H,G,H2,G2);  
    simdatasets{ii}     = simdata;   
    simregimesets{ii}   = simregimes;

    % -- Conditional Simulations -- %
    [simdata,regimes] = simulateddata_SV_restricted_shocks(GLOBALS,qm,Pss,xss,yss,H,G,H2,G2);
    simdatasets_shocks{ii} = simdata;

end
simdatafull     = cell2mat(simdatasets);
simregimesfull  = cell2mat(simregimesets');



%% -- Figure 2: The Logistic Functions and the Distributions of Their Arguments -- %%

ggamma01        = exp(qm(GLOBALS.parm_ind.loggamma01));
ggamma11        = exp(qm(GLOBALS.parm_ind.loggamma11)); 


% -- Plot of Logistics -- %
figure(2)
subplot(2,1,1)
    hold on
    yyaxis right
    [f,xi]=ksdensity(simdatafull(simregimesfull<3,GLOBALS.nx+GLOBALS.var_ind.y.bstar));
    h1 = plot(xi,f,'-k','LineWidth',2);
    set(gca,{'ycolor'},{'k'})
    set(gca,'yticklabel',num2str(get(gca,'ytick')','%.1f'))
    ylabel('Density')
    yyaxis left
    xi_step = (max(xi)-min(xi))/(10000-1);
    x = min(xi):xi_step:max(xi);
    y1  =  exp(-ggamma01*x) ./ (1 + exp(-ggamma01*x));
    h2=plot(x, y1(1,:),'--r','LineWidth',1.5);
    plot([0 0],[0 1],'k','LineWidth',1.5)
    set(gca,{'ycolor'},{'k'})
    yticks(0:0.25:1)
    legend([h1 h2],'Dist. of Borrowing Cushion (right)','Probability of Binding in t+1 (left)','Location','Northeast');
    hold off
    xlim([-10 20])
    xlabel('B^{\ast}_{t}')
    ylabel('Probability')
    title('Borrowing Cushion and Transition Probability in Non-Binding Regime')
subplot(2,1,2)
    hold on
    yyaxis right
    [f,xi]=ksdensity(simdatafull(simregimesfull>2,GLOBALS.nx+GLOBALS.var_ind.y.llambda));
    h1 = plot(xi,f,'-k','LineWidth',2);
    set(gca,{'ycolor'},{'k'})
    set(gca,'yticklabel',num2str(get(gca,'ytick')','%.1f'))
    ylabel('Density')
    yyaxis left
    xi_step = (max(xi)-min(xi))/(10000-1);
    x = min(xi):xi_step:max(xi);
    y2  = exp(-ggamma11*x) ./ (1 + exp(-ggamma11*x));
    h2 = plot(x, y2(1,:),'--r','LineWidth',1.5);
    plot([0 0],[0 1],'k','LineWidth',1.5)
    set(gca,{'ycolor'},{'k'})
    xlim([-0.1 0.1])
    yticks(0:0.25:1)
    ylabel('Probability')
    xlim([-0.05 0.05])
    legend([h1 h2],'Dist. of Multiplier (right)','Probability of Non-Binding in t+1 (left)','Location','Northeast');
    hold off
    xlabel('\lambda_{t}')
    title('Multiplier and Transition Probability in Binding Regime')
print('-dpdf', '..\Figures and Tables\Figure_2.pdf', '-fillpage')



%% -- Table 4: Simulated Second Moments: Data and Model -- %%

% -- Loading Data -- %
load('../Data/bfor_data');

y = [ygrow...
    cgrow...
    igrow...
    rint...
    cay...
    tby ...
    invtot]';


tab = NaN(size(y,1),6);


% -- Standard Deviations -- %
tab(:,1) = 100*std(y,1,2);
tab(:,3) = tab(:,1)./tab(1,1);

% -- Correlation with GDP -- %
for ii = 1:size(y,1)
    tab(ii,5) = corr(ygrow,y(ii,:)');
end


% -- Moments from Simulated Data -- %
model_stds = NaN(size(simdatasets,1),size(y,1));
model_corr = NaN(size(simdatasets,1),size(y,1));
model_auto = NaN(size(simdatasets,1),size(y,1));
for ii = 1:size(simdatasets,1)
    simdata = simdatasets{ii};

    y_sim = [
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.output) - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.output) + simdata(2:end,GLOBALS.var_ind.x.z) ...
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.cons)   - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.cons)   + simdata(2:end,GLOBALS.var_ind.x.z) ...
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.invest) - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.invest) + simdata(2:end,GLOBALS.var_ind.x.z) ...
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.intrate) ...
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.caratio)  ...
      simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.tbyratio)  ...
      simdata(2:end,GLOBALS.var_ind.x.tot) - simdata(1:end-1,GLOBALS.var_ind.x.tot)
    ]';

    model_stds(ii,:)  = std(y_sim,1,2);
    for jj = 1:size(y,1)
        model_corr(ii,jj)  = corr(y_sim(1,:)',y_sim(jj,:)');
        model_auto(ii,jj)  = corr(y_sim(jj,2:end)',y_sim(jj,1:end-1)');
    end
   
end
tab(:,2) = 100*mean(model_stds);
tab(:,4) = mean(model_stds)./mean(model_stds(:,1));
tab(:,6) = mean(model_corr);



% -- Table of Moments -- %
simtablelist=    { 'Output Growth';
                    'Consumption Growth';
                    'Investment Growth';
                    'Country Interest Rate';
                    'Current Account to GDP Ratio';
                    'Trade Balance to GDP Ratio';
                    'Import Price Growth'
                    };


disp(' ')
delete ../Figures and Tables/Table_4.tex;
diary('../Figures and Tables/Table_4.tex');
for ii = 1:size(y,1)
    latexPrt(simtablelist{ii},tab(ii,:),2)
end
diary off;

%% -- Table 5: Variance Decomposition -- %%
model_vd    = NaN(size(simdatasets,1),5,GLOBALS.ne);
vd_labels   = {'GDP Growth','Cons Growth','Inv Growth','TB/GDP','CA/GDP'};


for ii = 1:size(simdatasets,1)
    
    for jj = 1:GLOBALS.ne

        simdata = simdatasets_shocks{ii}{jj};

        y_sim = [
            simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.output) - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.output) + simdata(2:end,GLOBALS.var_ind.x.z) ...
            simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.cons)   - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.cons)   + simdata(2:end,GLOBALS.var_ind.x.z) ...
            simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.invest) - simdata(1:end-1,GLOBALS.nx+GLOBALS.var_ind.y.invest) + simdata(2:end,GLOBALS.var_ind.x.z) ...
            simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.tbyratio) ...
            simdata(2:end,GLOBALS.nx+GLOBALS.var_ind.y.caratio)
            ]';
    
        model_vd(ii,:,jj)  = std(y_sim,1,2);

    end

end


disp(' ')
delete ../Figures and Tables/Table_5.tex;
diary('../Figures and Tables/Table_5.tex');
tab = squeeze(mean(model_vd,1));
tab = 100*tab./sum(tab,2);
for ii = 1:5
    latexPrt(vd_labels{ii},tab(ii,:),2)
end
diary off;




  
    

    
